/*

	Copyright (c) 2004 PXI Project Team

	Permission is hereby granted, free of charge, to any person obtaining a copy of
	this software and associated documentation files (the "Software"), to deal in the
	Software without restriction, including without limitation the rights to use,
	copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
	Software, and to permit persons to whom the Software is furnished to do so,
	subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all copies
	or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
	INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
	PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
	OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

/*

	CLASS variant
 
	The variant class is used to hold a value that may be represented by multiple data types.
	Its job is to store the value internally, and convert between the various types as necessary.
	Obviously, you want this to be as quick as possible, and the functions tend to be fairly
	short, so it tends to do a lot of inlining.
 
	RETROFIT: We'll be using variants to store nonscalar values (such as objects) because the
	architecture is already equipped to handle variants, meaning we don't have to modify anything
	to be able to push our custom values around. It is critical that we not attempt to use such
	'packed' variants as scalar values, or we will most likely be rewarded with a system failure.
 
	TYPICAL USAGE
 
	Generally, you create variant with one of its two primary data types - a string or a double.
	You just use normal assignment to add a value, and let the object figure out how to store it.
	If you want a number out of it, you call intVal(), if you want a string out of it, you call
	strVal(). Note that it will blow your program to kingdom come if it can't make the conversion.
 
	The class is self-cleaning.

*/

#ifndef _____VARIANT_____
#define _____VARIANT_____

#include <string>
#include <map>
#include "chunk.h"
#include "PXIMutableObject.h"
using namespace std;

class variant
{
public:
								variant				( const double& = 0 );
								variant				( const string& );
								variant				( const variant& );
								variant				( const PXIMutableObject& );
								~variant			( void );

inline const	variant &		Copy				( const variant& );
inline const	variant &		Copy				( const string& );
inline const	variant &		Copy				( const double& );

inline const	double &		intVal				( );
inline const	string &		strVal				( );

inline			void			setDict				( const string&, const variant& );
inline const	variant			getDict				( const string& );

inline const	variant &		operator=			( const variant& );
inline const	variant &		operator=			( const string& );
inline const	variant &		operator=			( const double& );
inline const	PXIMutableObject& operator=			( const PXIMutableObject& );

				chunk			*chunkValue;
inline const	bool			isDirty				( void );
inline const	bool			isObject			( void );

private:
				double			longValue;
				string			stringValue;
				
				map< string,
					variant>   *dictValue;

				int				hbobjectvalue; /* Unimplemented */
	PXIMutableOBject			objectValue;
				bool			vtype;
				bool			dirty;
				bool			obj;
};

inline const variant &variant::Copy( const variant &v )
{
    if ( v.vtype )
    {
        vtype = true;
        longValue = v.longValue;
    }
    else
    {
        vtype = false;
        stringValue = v.stringValue;
    }
	dirty = true;
    return *this;
}

inline const variant &variant::Copy( const string &s )
{
    vtype = false;
    stringValue = s;
	dirty = true;
    return *this;
}

inline const variant &variant::Copy( const double &l )
{
    vtype = true;
    longValue = l;
	dirty = true;
    return *this;
}

inline const variant &variant::Copy( const PXIMutableObject &p )
{
	objectValue = p;
	dirty = true;
	return * this;
}

inline const double &variant::intVal( void )
{
    char **a = NULL;
    if ( vtype )
    {
        return longValue;
    }
    else
    {
        vtype = true;
        longValue = strtod( stringValue.c_str(), a );
        return longValue;
    }
}

inline const string &variant::strVal( void )
{
    char a[ 255 ] = {0};
    if ( vtype )
    {
        vtype = false;
        snprintf( a, 254, "%g", longValue );
        stringValue = a;
        return stringValue;
    }
    else
    {
        return stringValue;
    }
}

inline const variant &variant::operator=( const variant &v )
{
    return Copy( v );
}

inline const variant &variant::operator=( const string &v )
{
    return Copy( v );
}

inline const variant &variant::operator=( const double &v )
{
    return Copy( v );
}

inline const variant &variant::operator=( const PXIMutableOBject &v )
{
    return Copy( v );
}

inline void variant::setDict( const string &s, const variant &v )
{
	if( ! dictValue )
	{
		dictValue = new map< string, variant >;
	}
	( * dictValue ).insert( pair< string, variant >( s, v ) );
}

inline const variant variant::getDict( const string &s )
{
	string s1;
	if( ! dictValue )
	{
		dictValue = new map< string, variant>;
		return variant( s1 );
	}
	if( ( * dictValue ).find( s ) == ( * dictValue ).end() )
	{
		( * dictValue ).insert( pair< string, variant>( s, variant ( s ) ) );
		return variant( s1 );
	}
	else
	{
		return ( * ( * dictValue ).find( s ) ).second;
	}
}

inline const bool variant::isDirty( void )
{
	return dirty;
}

inline const bool variant::isObject( void )
{
	return obj;
}

#endif
